/*
    Copyright  1998-2005, The AROS Development Team. All rights reserved. 
    $Id: clipboard.c,v 1.9 2012/11/27 13:01:40 krashan Exp $

    Clipboard device.
*/

/****************************************************************************************/

#define VERSION 50
#define REVISION 8
#define S(x) #x
#define VSTRING "$VER: clipboard device " S(VERSION) "." S(REVISION) "(27.11.2012)\r\n"


#define AROS_ALMOST_COMPATIBLE 1
#define USE_INLINE_STDARG
#include <exec/resident.h>
#include <devices/clipboard.h>
#include <devices/newstyle.h>
#include <exec/io.h>
#include <exec/initializers.h>
#include <libraries/iffparse.h>
#include <hardware/byteswap.h>
#include <proto/exec.h>
#include <proto/dos.h>
#include <proto/utility.h>
#include <clib/alib_protos.h>
#include <exec/memory.h>
#include <exec/errors.h>
#include <exec/lists.h>
#include <aros/libcall.h>
#include <aros/asmcall.h>
#include "clipboard_intern.h"
#define DEBUG 0
#include <aros/debug.h>
#include <aros/machine.h>
//#include <stddef.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef __MORPHOS__
unsigned long __abox__ = 1;
#endif

/****************************************************************************************/
#ifndef __MORPHOS__
#define NEWSTYLE_DEVICE 1
#endif

#define ioClip(x)	((struct IOClipReq *)x)

#define WRITEBUFSIZE	4096
#define MAXCLIPSIZE	0x7fffffff

/****************************************************************************************/

static void readCb(struct IORequest *ioreq, struct ClipboardBase *CBBase);
static void writeCb(struct IORequest *ioreq, struct ClipboardBase *CBBase);
static void updateCb(struct IORequest *ioreq, struct ClipboardBase *CBBase);

/****************************************************************************************/

static const char name[];
static const char version[];
static const APTR inittabl[4];
static void *const functable[];
static const UBYTE datatable;

struct  ClipboardBase  *AROS_SLIB_ENTRY(init, Clipboard)();
void  AROS_SLIB_ENTRY(open, Clipboard)();
BPTR  AROS_SLIB_ENTRY(close, Clipboard)();
BPTR  AROS_SLIB_ENTRY(expunge, Clipboard)();
int   AROS_SLIB_ENTRY(null, Clipboard)();
void  AROS_SLIB_ENTRY(beginio, Clipboard)();
LONG  AROS_SLIB_ENTRY(abortio, Clipboard)();

/****************************************************************************************/

int AROS_SLIB_ENTRY(entry, Clipboard)(void)
{
	/* If the device was executed by accident return error code. */
	return -1;
}

const struct Resident Clipboard_resident =
{
	RTC_MATCHWORD,
	(struct Resident *)&Clipboard_resident,
	(APTR)(&Clipboard_resident + 1),
#ifdef __MORPHOS__
	RTF_PPC | RTF_EXTENDED | RTF_AUTOINIT,
#else
	RTF_AUTOINIT|RTF_COLDSTART,
#endif
	VERSION,
	NT_DEVICE,
	45,
	(char *)name,
	(char *)&version[6],
	(ULONG *)inittabl,
#ifdef __MORPHOS__
	REVISION,	/* Revision */
	NULL /* Tags */
#endif
};

static const char name[] = "clipboard.device";

static const char version[] = VSTRING;

static const APTR inittabl[4] =
{
	(APTR)sizeof(struct ClipboardBase),
	(APTR)functable,
#ifdef __MORPHOS__
	NULL,
#else
	(APTR)&datatable,
#endif
	&AROS_SLIB_ENTRY(init, Clipboard)
};

static void *const functable[] =
{
#ifdef __MORPHOS__
	(void *const) FUNCARRAY_32BIT_NATIVE,
#endif
	&AROS_SLIB_ENTRY(open, Clipboard),
	&AROS_SLIB_ENTRY(close, Clipboard),
	&AROS_SLIB_ENTRY(expunge, Clipboard),
	&AROS_SLIB_ENTRY(null, Clipboard),
	&AROS_SLIB_ENTRY(beginio, Clipboard),
	&AROS_SLIB_ENTRY(abortio, Clipboard),
	(void *)-1
};

/****************************************************************************************/

#if NEWSTYLE_DEVICE

static const UWORD SupportedCommands[] =
{
	CMD_READ,
	CMD_WRITE,
	CMD_UPDATE,
	CBD_CHANGEHOOK,
	CBD_POST,
	CBD_CURRENTREADID,
	CBD_CURRENTWRITEID,
	NSCMD_DEVICEQUERY,
	0
};

#endif

/****************************************************************************************/

#ifdef __MORPHOS__
struct ClipboardBase *LIB_init(struct ClipboardBase *CBBase, BPTR segList, struct ExecBase *sysBase)
#else
AROS_UFH3(struct ClipboardBase *,  AROS_SLIB_ENTRY(init,Clipboard),
 AROS_UFHA(struct ClipboardBase *, CBBase, D0),
 AROS_UFHA(BPTR,          segList, A0),
 AROS_UFHA(struct ExecBase *, sysBase, A6))
#endif
{
	AROS_USERFUNC_INIT

	/* Store arguments */
	CBBase->cb_sysBase = sysBase;
	CBBase->cb_seglist = segList;

	InitSemaphore(&CBBase->cb_SignalSemaphore);
	NEWLIST(&CBBase->cb_UnitList);

	return CBBase;

	AROS_USERFUNC_EXIT
}

/****************************************************************************************/
#ifndef __MORPHOS__
/*this trick can't work with MorphOS since we can't rely on varargs to be a linear stream*/

/* Putchar procedure needed by RawDoFmt() */

AROS_UFH2(void, putchr,
          AROS_UFHA(UBYTE,    chr, D0),
          AROS_UFHA(STRPTR *, p,   A3))
{
	AROS_USERFUNC_INIT
	*(*p)++=chr;
	AROS_USERFUNC_EXIT
}

/****************************************************************************************/

#define cb_sprintf(CBBase, buffer, format, ...) \
({ ULONG _args[]={__VA_ARGS__}; APTR bufptr = buffer; RawDoFmt(format, _args, (VOID_FUNC)AROS_ASMSYMNAME(putchr), &bufptr); })

#endif


/****************************************************************************************/

AROS_LH3(void, open,
         AROS_LHA(struct IORequest *, ioreq, A1),
         AROS_LHA(ULONG,              unitnum, D0),
         AROS_LHA(ULONG,              flags, D1),
         struct ClipboardBase *, CBBase, 1, Clipboard)
{
	AROS_LIBFUNC_INIT

	ULONG opencnt;

	/* Keep compiler happy */
	(void) flags;

	ObtainSemaphore(&CBBase->cb_SignalSemaphore);

	/* Bump the opencnt here so that expunge can't screw us up. - Piru */

	Forbid();
	opencnt = CBBase->cb_device.dd_Library.lib_OpenCnt;
	CBBase->cb_device.dd_Library.lib_OpenCnt++;
	CBBase->cb_device.dd_Library.lib_Flags &= ~LIBF_DELEXP;
	Permit();

	/* Default: Open failed */
	ioClip(ioreq)->io_Error = IOERR_OPENFAIL;

	do
	{
		struct ClipboardUnit *unit;

		D(bug("clipboard.device/open: ioreq 0x%08lx unitnum %lu flags 0x%08lx\n",
		      ioreq, unitnum, flags));

#ifndef __MORPHOS__
#warning "You shouldn't check this..only leads to trouble"
#warning "Eh, if the iorequest is too small things will NUKE - Piru"
		if (ioreq->io_Message.mn_Length < sizeof(struct IOClipReq))
		{
			D(bug("clipboard.device/open: IORequest structure passed to OpenDevice is too small!\n"));
			break;
		}
#endif

		if (!CBBase->cb_DosBase)
		{
			CBBase->cb_DosBase = OpenLibrary("dos.library", 39);
			if (!CBBase->cb_DosBase)
			{
				break;
			}
		}

		if (!CBBase->cb_UtilityBase)
		{
			CBBase->cb_UtilityBase = OpenLibrary("utility.library", 36);
			if (!CBBase->cb_UtilityBase)
			{
				break;
			}
		}

		/* Set up clipboard directory if we are the first opener */

		if (opencnt == 0)
		{
			BPTR lock;

			D(bug("clipboard.device/Checking for CLIPS:\n"));

			lock = Lock("CLIPS:", ACCESS_READ);
			if (!lock)
			{
				/* CLIPS: is not assigned - revert to RAM:Clipboards */

				D(bug("clipboard.device/CLIPS: not found\n"));
				D(bug("clipboard.device/Checking for RAM:\n"));

				lock = Lock("RAM:", ACCESS_READ);
				if (!lock)
				{
					D(bug("clipboard.device/RAM: Not found. FAIL!"));
					break;
				}
				else
				{
					D(bug("clipboard.device/Found RAM:\n"));
					D(bug("clipboard.device/Checking for RAM:clipboards\n"));

					UnLock(lock);

					CBBase->cb_ClipDir = "RAM:clipboards/";

					lock = Lock("RAM:clipboards/", ACCESS_READ);
					if (!lock)
					{
						D(bug("clipboard.device/Not found -- creating RAM:Clipboards.\n"));
						lock = CreateDir("RAM:clipboards");
						if (!lock)
						{
							D(bug("clipboard.device/can't create RAM:clipboards dir, fallback to RAM:\n"));
							CBBase->cb_ClipDir = "RAM:";
						}
					}
				}
			}
			else
			{
				D(bug("clipboard.device/Found CLIPS:\n"));
				CBBase->cb_ClipDir = "CLIPS:";
			}

			/* Release the possible lock we have made */
			if (lock)
			{
				UnLock(lock);
			}
		}

		ForeachNode(&CBBase->cb_UnitList, unit)
		{
			D(bug("clipboard.device/ UnitNode 0x%08lx Unit %lu\n",
			      unit, unit->cu_Head.cu_UnitNum));

			if (unit->cu_Head.cu_UnitNum == unitnum)
			{
				D(bug("clipboard.device/ found UnitNode\n"));

				break;
			}
		}

		/* Unit already exists ? */
		if (unit->cu_Head.cu_Node.ln_Succ)
		{
			ioreq->io_Unit = (struct Unit *) unit;
			unit->cu_OpenCnt++;

			ioClip(ioreq)->io_Error = 0;

			D(bug("clipboard.device/Open successful, return 0\n"));
			break;
		}

		D(bug("clipboard.device/Building unit...\n"));

		unit = (struct ClipboardUnit *)AllocMem(sizeof(struct ClipboardUnit),
		                                        MEMF_CLEAR | MEMF_PUBLIC);
		if (unit)
		{
			ioreq->io_Unit = (struct Unit *) unit;

			unit->cu_Head.cu_UnitNum = unitnum;
			//unit->cu_WriteID = 0;
			//unit->cu_PostID = 0;
			unit->cu_OpenCnt = 1;

			NEWLIST((struct List *) &unit->cu_PostRequesters);
			NEWLIST((struct List *) &unit->cu_HookList);
			InitSemaphore(&unit->cu_UnitLock);
			InitSemaphore(&unit->cu_HookListLock);

			/* Construct clipboard unit filename. */
			#ifdef __MORPHOS__
			NewRawDoFmt("%s%lu", NULL, unit->cu_clipFilename, (ULONG) CBBase->cb_ClipDir, unitnum);
			#else
			cb_sprintf(CBBase, unit->cu_clipFilename, "%s%lu", (ULONG) CBBase->cb_ClipDir,
			           unitnum);
			#endif

			unit->cu_Satisfy.sm_Unit = unitnum;

			/* Initialization is done, and everything went OK. Add unit to the
			   list of clipboard units. */
			ADDHEAD((struct List *)&CBBase->cb_UnitList, (struct Node *)unit);

			/* Check if there is already a clipboard file for this unit existing.
			   If yes, then set WriteID to 1 so that CMD_READing works, and
			   also setup clipSize */

			unit->cu_clipFile = Open(unit->cu_clipFilename, MODE_OLDFILE);
			if (unit->cu_clipFile)
			{
				if (Seek(unit->cu_clipFile, 0, OFFSET_END) != -1)
				{
					unit->cu_clipSize = Seek(unit->cu_clipFile, 0, OFFSET_BEGINNING);

					D(bug("clipboard.device/ <%s> clipsize %ld\n",
					      unit->cu_clipFilename, unit->cu_clipSize));
					if (unit->cu_clipSize != (ULONG)-1)
					{
						D(bug("clipboard.device/ WriteID set\n"));
						unit->cu_WriteID = 1;
					}
				}
				Close(unit->cu_clipFile);
				unit->cu_clipFile = 0;
			}
			else
			{
				D(bug("clipboard.device/no <%s> file\n", unit->cu_clipFilename));
			}

			D(bug("clipboard.device/Open successful, return 0\n"));
			ioreq->io_Error = 0;
		}
		else
		{
			D(bug("clipboard.device/Couldn't alloc Unit\n"));
		}

	} while (0);

	if (ioreq->io_Error)
	{
		ioreq->io_Unit = (struct Unit *) -1;
		Forbid();
		CBBase->cb_device.dd_Library.lib_OpenCnt--;
		Permit();
	}

	ReleaseSemaphore(&CBBase->cb_SignalSemaphore);

	AROS_LIBFUNC_EXIT
}

/****************************************************************************************/

AROS_LH1(BPTR, close,
         AROS_LHA(struct IORequest *,     ioreq,  A1),
         struct ClipboardBase *, CBBase,  2, Clipboard)
{
	AROS_LIBFUNC_INIT

	#undef SysBase
	struct ExecBase *SysBase = CBBase->cb_sysBase; /* local SysBase ptr for flush */
	struct ClipboardUnit *unit = (APTR) ioreq->io_Unit;

	BPTR ret = 0;   /* Temporary variable to preserve seglist. */

	D(bug("clipboard.device/close: ioreq 0x%08lx\n", ioreq));

	/* Let any following attemps to use the device crash hard. */
	ioreq->io_Device = (struct Device *) -1;

	ObtainSemaphore(&CBBase->cb_SignalSemaphore);

	unit->cu_OpenCnt--;

	D(bug("clipboard.device/close: unitcnt %lu\n", unit->cu_OpenCnt));

	if (unit->cu_OpenCnt == 0)
	{
		D(bug("clipboard.device/close: removeunit\n", ioreq));
		REMOVE((struct Node *)ioreq->io_Unit);

		/* If the file is still open here, some app only did partial
		 * CMD_READ or CMD_WRITE without CMD_UPDATE. Better Close() the
		 * file manually in this case. - Piru
		 */
		if (unit->cu_clipFile)
		{
			D(bug("clipboard.device/close: Close pending clipfile 0x%08lx\n",
			      unit->cu_clipFile));

			Close(unit->cu_clipFile);
			unit->cu_clipFile = 0;
		}

		FreeMem(ioreq->io_Unit, sizeof(struct ClipboardUnit));

		/* Let any following attemps to use the device crash hard. */
		ioreq->io_Unit = (struct Unit *) -1;
	}

	Forbid();

	if (--CBBase->cb_device.dd_Library.lib_OpenCnt == 0)
	{
		if (CBBase->cb_device.dd_Library.lib_Flags & LIBF_DELEXP)
		{
			REG_A6 = (ULONG) CBBase;
			ret = expunge();
		}
	}

	if (!ret)
	{
		ReleaseSemaphore(&CBBase->cb_SignalSemaphore);
	}

	Permit();

	return ret;

	#define SysBase CBBase->cb_sysBase

	AROS_LIBFUNC_EXIT
}

/****************************************************************************************/

AROS_LH0(BPTR, expunge, struct ClipboardBase *, CBBase, 3, Clipboard)
{
	AROS_LIBFUNC_INIT

	BPTR ret;   /* Temporary variable to preserve seglist. */

	D(bug("clipboard.device/expunge:\n"));

	Forbid();

	if (CBBase->cb_device.dd_Library.lib_OpenCnt)
	{
		/* Do not expunge the device. Set the delayed expunge flag and
		   return. */
		CBBase->cb_device.dd_Library.lib_Flags |= LIBF_DELEXP;

		Permit();

		return 0;
	}

	/* Get rid of the device. Remove it from the list. */
	REMOVE(&CBBase->cb_device.dd_Library.lib_Node);

	Permit();

	/* Get returncode here - FreeMem() will destroy the field. */
	ret = CBBase->cb_seglist;

	CloseLibrary(CBBase->cb_UtilityBase);
	CloseLibrary(CBBase->cb_DosBase);

	/* Free the memory. */
	FreeMem((char *)CBBase - CBBase->cb_device.dd_Library.lib_NegSize,
	        CBBase->cb_device.dd_Library.lib_NegSize
	        + CBBase->cb_device.dd_Library.lib_PosSize);

	D(bug("clipboard.device/expunge: done\n"));

	return ret;

	AROS_LIBFUNC_EXIT
}

/****************************************************************************************/

AROS_LH0(int, null, struct ClipboardBase *, CBBase, 4, Clipboard)
{
	AROS_LIBFUNC_INIT
	D(bug("clipboard.device/null:\n"));
	(void) CBBase;
	return 0;
	AROS_LIBFUNC_EXIT
}

/****************************************************************************************/

AROS_LH1(void, beginio,
         AROS_LHA(struct IORequest *, ioreq, A1),
         struct ClipboardBase *, CBBase, 5, Clipboard)
{
	AROS_LIBFUNC_INIT
	struct ClipboardUnit *unit = (APTR) ioreq->io_Unit;

	ioreq->io_Error = 0;

	switch (ioreq->io_Command)
	{
#if NEWSTYLE_DEVICE
		case NSCMD_DEVICEQUERY:
			if (ioClip(ioreq)->io_Length < ((LONG)OFFSET(NSDeviceQueryResult, SupportedCommands)) + sizeof(UWORD *))
			{
				ioreq->io_Error = IOERR_BADLENGTH;
			}
			else
			{
				struct NSDeviceQueryResult *d;

				d = (struct NSDeviceQueryResult *)ioClip(ioreq)->io_Data;

				d->DevQueryFormat    = 0;
				d->SizeAvailable     = sizeof(struct NSDeviceQueryResult);
				d->DeviceType        = NSDEVTYPE_CLIPBOARD;
				d->DeviceSubType     = 0;
				d->SupportedCommands = (UWORD *)SupportedCommands;

				ioClip(ioreq)->io_Actual = sizeof(struct NSDeviceQueryResult);
			}
			break;
#endif

		case CBD_CHANGEHOOK:

			D(bug("clipboard.device/Command: CBD_CHANGEHOOK\n"));

			ObtainSemaphore(&unit->cu_HookListLock);

			/* io_Length is used as a means of specifying if the hook
			   should be added or removed. */
			switch (ioClip(ioreq)->io_Length)
			{
				case 0:
					REMOVE((struct Node *)(ioClip(ioreq)->io_Data));
					break;

				case 1:
					ADDHEAD((struct List *)&unit->cu_HookList,
					        (struct Node *)ioClip(ioreq)->io_Data);
					break;

				default:
					ioreq->io_Error = IOERR_BADLENGTH;
					break;
			}
			ReleaseSemaphore(&unit->cu_HookListLock);
			break;

		case CMD_WRITE:

			D(bug("clipboard.device/Command: CMD_WRITE\n"));

			writeCb(ioreq, CBBase);
			break;

		case CMD_READ:

			D(bug("clipboard.device/Command: CMD_READ\n"));

			/* Get new ID if this is the beginning of a read operation */
			if (ioClip(ioreq)->io_ClipID == 0)
			{
				D(bug("clipboard.device/CMD_READ: Trying to get unit lock. Calling ObtainSemaphore [me=0x%08lx].\n", FindTask(NULL)));

				ObtainSemaphore(&unit->cu_UnitLock);

				D(bug("clipboard.device/CMD_READ: Got unit lock.\n"));

				/* If the last write was actually a POST, we must tell
				   the POSTer to WRITE the clip immediately, and we
				   will wait until he have done so. Then we check
				   again in case somebody managed to sneak in a
				   CBD_POST after the CMD_UPDATE. */

				while (unit->cu_WriteID != 0 &&
				       unit->cu_WriteID == unit->cu_PostID)
				{
					struct PostRequest pr;

					pr.pr_Waiter = FindTask(NULL);

					/* Make sure we are signalled. */
					ADDTAIL((struct List *) &unit->cu_PostRequesters, (struct Node *) &pr);

					/* A poster reading will deadlock that process
					 * until somebody else writes to the
					 * clipboard. AmigaOS behaves exactly the same so
					 * it's ok. It's just plain stupid anyway. */

					if (unit->cu_PostPort)
					{
						D(bug("clipboard.device/Command: CMD_READ..notify PostPort 0x%08lx\n", unit->cu_PostPort));

						unit->cu_Satisfy.sm_ClipID = unit->cu_PostID;
						PutMsg(unit->cu_PostPort, (struct Message *)&unit->cu_Satisfy);
						unit->cu_PostPort = NULL;
					}
					else
					{
						D(bug("clipboard.device/Command: no PostPort [me=0x%08lx]\n", FindTask(NULL)));
					}

					Forbid();
					ReleaseSemaphore(&unit->cu_UnitLock);
					SetSignal(0, SIGF_SINGLE);
					Wait(SIGF_SINGLE);
					Permit();
					D(bug("Got SIGF_SINGLE [me=0x%08lx]\n", FindTask(NULL)));
					ObtainSemaphore(&unit->cu_UnitLock);
					D(bug("Got semaphore[me=0x%08lx]\n", FindTask(NULL)));

					if (pr.pr_Link.mln_Succ->mln_Succ != NULL)
					{
						/* Wake up next reader */
						Signal(((struct PostRequest*) pr.pr_Link.mln_Succ)->pr_Waiter, SIGF_SINGLE);
					}

					REMOVE((struct Node *) &pr);
				}

				unit->cu_ReadID++;
				ioClip(ioreq)->io_ClipID = unit->cu_ReadID;

				unit->cu_clipFile = Open(unit->cu_clipFilename, MODE_OLDFILE);

				if (!unit->cu_clipFile)
				{
					D(bug("clipboard.device/CMD_READ: No clip file. Calling ReleaseSemaphore [me=0x%08lx]\n", FindTask(NULL)));
					ReleaseSemaphore(&unit->cu_UnitLock);
					ioClip(ioreq)->io_ClipID = -1;
					ioClip(ioreq)->io_Actual = 0;
//					ioClip(ioreq)->io_Error = IOERR_ABORTED;
					break;
				}
			}
			else if (ioClip(ioreq)->io_ClipID != unit->cu_ReadID)
			{
				D(bug("clipboard.device/CMD_READ: Invalid clip id.\n"));
				ioClip(ioreq)->io_Actual = 0;
//				ioClip(ioreq)->io_Error = IOERR_ABORTED;
				break;
			}

			D(bug("LCS var her. ;)\n"));
			readCb(ioreq, CBBase);

			break;


		case CMD_UPDATE:
			D(bug("clipboard.device/Command: CMD_UPDATE\n"));

			updateCb(ioreq, CBBase);
			break;


		case CBD_POST:
			D(bug("clipboard.device/Command: CBD_POST [me=0x%08lx]\n", FindTask(NULL)));
			ObtainSemaphore(&unit->cu_UnitLock);

			unit->cu_WriteID++;
			unit->cu_PostID   = unit->cu_WriteID;
			unit->cu_PostPort = (struct MsgPort *)ioClip(ioreq)->io_Data;

			ioClip(ioreq)->io_ClipID = unit->cu_PostID;

			ReleaseSemaphore(&unit->cu_UnitLock);

			D(bug("clipboard.device/CBD_POST: Calling monitoring hooks\n"));

			/* Call monitoring hooks. */
			{
				struct Hook        *tnode;
				struct ClipHookMsg  chmsg;

				chmsg.chm_Type      = 0;
				chmsg.chm_ChangeCmd = CBD_POST;
				chmsg.chm_ClipID    = unit->cu_PostID;

				ObtainSemaphore(&unit->cu_HookListLock);

				ForeachNode(&unit->cu_HookList, tnode)
				{
					D(bug("Calling hook 0x%08x\n", tnode));
					CallHookA(tnode, unit, &chmsg);
				}
				D(bug("Done\n"));

				ReleaseSemaphore(&unit->cu_HookListLock);
			}

			D(bug("clipboard.device/CBD_POST: Called monitoring hooks\n"));		    

#if 0
			// This does not seem to be robust enough; it can lead to
			// a ping-pong effect. Never mind then.

			ObtainSemaphore(&unit->cu_UnitLock);

			if (!IsListEmpty((struct List *) &unit->cu_PostRequesters))
			{
				/* Normally, this can never happen. However, if an app
				   deadlocked by posting and then reading, try to make
				   this CBD_POST turn into a CMD_WRITE immediately. */

				D(bug("clipboard.device/Command: CMD_POST..notify PostPort 0x%08lx\n", unit->cu_PostPort));

				unit->cu_Satisfy.sm_ClipID = unit->cu_PostID;
				PutMsg(unit->cu_PostPort, (struct Message *)&unit->cu_Satisfy);
				unit->cu_PostPort = NULL;
			}

			ReleaseSemaphore(&unit->cu_UnitLock);
#endif
			break;


		case CBD_CURRENTREADID:
			D(bug("clipboard.device/Command: CBD_CURRENTREADID\n"));
			ioClip(ioreq)->io_ClipID = unit->cu_WriteID;
			/* NOTE: NOT A BUG! Was PostID. Note that AmigaOS really has a
			   ReadID counter that is *almost* always the same as WriteID. */
			break;


		case CBD_CURRENTWRITEID:
			D(bug("clipboard.device/Command: CBD_CURRENTWRITEID\n"));
			ioClip(ioreq)->io_ClipID = unit->cu_WriteID;
			break;


		default:
			D(bug("clipboard.device/Command: <UNKNOWN> (%lu = 0x%04lx)\n", ioreq->io_Command, ioreq->io_Command));
			ioreq->io_Error = IOERR_NOCMD;
			break;

	} /* switch (ioreq->io_Command) */

	/* If the quick bit is not set, send the message to the port */
	if (!(ioreq->io_Flags & IOF_QUICK))
	{
		ReplyMsg(&ioreq->io_Message);
	}

	AROS_LIBFUNC_EXIT
}

/****************************************************************************************/

AROS_LH1(LONG, abortio,
         AROS_LHA(struct IORequest *,      ioreq, A1),
         struct ClipboardBase *, CBBase, 6, Clipboard)
{
	AROS_LIBFUNC_INIT

	/* Keep compiler happy */
	(void) ioreq;
	(void) CBBase;

	D(bug("clipboard.device/abortio: ioreq 0x%08lx\n", ioreq));
	/* Nothing to abort */
	return 0;

	AROS_LIBFUNC_EXIT
}

/****************************************************************************************/

static void UnlockClip(struct ClipboardBase *CBBase, struct IOClipReq *r, struct ClipboardUnit *cu)
{
	Close(cu->cu_clipFile);
	cu->cu_clipFile = 0;
	ReleaseSemaphore(&cu->cu_UnitLock);
	r->io_Actual = 0;
	r->io_ClipID = -1;
}


static void readCb(struct IORequest *ioreq, struct ClipboardBase *CBBase)
{
	struct ClipboardUnit *unit = (APTR)ioreq->io_Unit;
	LONG block_length;

	/*-------------------------------*/
	/* Is there anything to be read? */
	/*-------------------------------*/

	if (unit->cu_WriteID == 0)
	{
		D(bug("clipboard.device/readcb: nothing to read. setting IOERR_ABORTED as error and releasing semaphore [me=0x%08lx]\n", FindTask(NULL)));
		UnlockClip(CBBase, ioClip(ioreq), unit);
//		ioClip(ioreq)->io_Error = IOERR_ABORTED;
		return;
	}

	/*-------------------------------------------------------------------------------------------------------------------------*/
	/* Independently of io_Data being NULL or not, if io_Offset is equal or greater than clip length, clip should be unlocked. */
	/*-------------------------------------------------------------------------------------------------------------------------*/

	if (ioClip(ioreq)->io_Offset >= unit->cu_clipSize)
	{
		D(bug("clipboard.device/readCb: io_Offset detected \"end of file\". Closing clipfile and releasing semaphore [me=0x%08lx]\n", FindTask(NULL)));
		UnlockClip(CBBase, ioClip(ioreq), unit);
		return;
	}

	/*-----------------------------------------------------------------------------------------------------------------------*/
	/* Reading. Note that code is the same for io_Data being NULL or not. For io_Data == NULL just Seek() and Read() are not */
	/* called. Also clip is never unlocked for io_Actual being higher than 0. Versions up to 50.7 including were buggy here. */
	/*-----------------------------------------------------------------------------------------------------------------------*/

	block_length = unit->cu_clipSize - ioClip(ioreq)->io_Offset;                                   // this is maximum we can read from clip
	if (block_length > ioClip(ioreq)->io_Length) block_length = ioClip(ioreq)->io_Length;          // but no more than really requested

	if (ioClip(ioreq)->io_Data)                                                                    // seek and read only if data are to be stored
	{
		D(bug("clipboard.device/readCb: Doing read Seek() at offset %lu.\n", ioClip(ioreq)->io_Offset));

		if ((Seek(unit->cu_clipFile, ioClip(ioreq)->io_Offset, OFFSET_BEGINNING) == -1) ||
		 ((LONG)(ioClip(ioreq)->io_Actual = Read(unit->cu_clipFile, ioClip(ioreq)->io_Data, block_length)) == -1))
		{
			D(bug("clipboard.device/readCb: Seek or Read error %ld\n", IoErr()));
			UnlockClip(CBBase, ioClip(ioreq), unit);
			ioreq->io_Error = IOERR_ABORTED;
		}
	}
	else
	{
		ioClip(ioreq)->io_Actual = block_length;
	}

	/*-----------------------------------------------------------------------------------------------------------*/
	/* In any case update offset. For Read/Seek fail, io_Actual has been set to zero, so offset does not change. */
	/*-----------------------------------------------------------------------------------------------------------*/

	ioClip(ioreq)->io_Offset += ioClip(ioreq)->io_Actual;
	return;
}

/****************************************************************************************/

static const UBYTE zero[WRITEBUFSIZE];

static ULONG writezero(struct IORequest *ioreq, ULONG len, struct ClipboardBase *CBBase)
{
	struct ClipboardUnit *unit = (APTR) ioreq->io_Unit;

	while (len)
	{
		LONG size = len > WRITEBUFSIZE ? WRITEBUFSIZE : len;
		LONG actual = Write(unit->cu_clipFile, (APTR)zero, size);
		if (actual != size)
		{
			break;
		}
		len -= actual;
	}
	return len;
}

/****************************************************************************************/

static void writeCb(struct IORequest *ioreq, struct ClipboardBase *CBBase)
{
	struct ClipboardUnit *unit = (APTR) ioreq->io_Unit;

	D(bug("clipboard.device/writeCb: Trying to get unit lock. Calling ObtainSemaphore [me=0x%08lx]\n", FindTask(NULL)));
	ObtainSemaphore(&unit->cu_UnitLock);
	D(bug("clipboard.device/writeCb: Got unit lock.\n"));

	if (ioClip(ioreq)->io_ClipID == 0 ||
	    ioClip(ioreq)->io_ClipID == unit->cu_PostID)
	{
		/* A new write begins... */

		unit->cu_clipSize = 0;

		if (ioClip(ioreq)->io_ClipID == 0)
		{
			unit->cu_WriteID++;
			ioClip(ioreq)->io_ClipID = unit->cu_WriteID;
		}

		/* No more POST writes accepted */
		unit->cu_PostID = 0;

		if (!(unit->cu_clipFile = Open(unit->cu_clipFilename, MODE_NEWFILE)))
		{
			D(bug("clipboard.device/writeCb: Opening clipfile in MODE_NEWFILE failed. Releasing Semaphore [me=0x%08lx]\n", FindTask(NULL)));
			ReleaseSemaphore(&unit->cu_UnitLock);
			ioClip(ioreq)->io_Error = IOERR_ABORTED;
			ioClip(ioreq)->io_Actual = 0;
			ioClip(ioreq)->io_ClipID = -1;
			return;
		}

		D(bug("clipboard.device/writeCb: Opened file %s\n", unit->cu_clipFilename));
	}
	else if (ioClip(ioreq)->io_ClipID == unit->cu_WriteID)
	{
		D(bug("We already have the semaphore. [me=0x%08lx]\n", FindTask(NULL)));
		ReleaseSemaphore(&unit->cu_UnitLock);

		/* Continue the previous write */
	}
	else
	{
		D(bug("Invalid ClipID. Releasing Semaphore [me=0x%08lx]\n", FindTask(NULL)));
		ReleaseSemaphore(&unit->cu_UnitLock);

		/* Error */
		ioClip(ioreq)->io_Error = IOERR_ABORTED;
		ioClip(ioreq)->io_Actual = 0;
		return;
	}

	Seek(unit->cu_clipFile, ioClip(ioreq)->io_Offset, OFFSET_BEGINNING);
	if (ioClip(ioreq)->io_Offset > unit->cu_clipSize)
	{
		ULONG len = ioClip(ioreq)->io_Offset - unit->cu_clipSize;

		len = writezero(ioreq, len, CBBase);

		if (len)
		{
			D(bug("clipboard.device/writeCb: write failed\n"));
			Close(unit->cu_clipFile);
			unit->cu_clipFile = 0;
			D(bug("clipboard.device/writeCb: releasing semaphore [me=0x%08lx]\n", FindTask(NULL)));
			ReleaseSemaphore(&unit->cu_UnitLock);
			ioClip(ioreq)->io_Error = IOERR_ABORTED;
			ioClip(ioreq)->io_Actual = 0;
			ioClip(ioreq)->io_ClipID = -1;
			return;
		}
	}

	D(bug("clipboard.device/writeCb: Did Seek(), offset = %lu\n", ioClip(ioreq)->io_Offset));
	D(bug("clipboard.device/Doing Write: data length = %lu  data = %02lx%02lx%02lx%02lx (%lc%lc%lc%lc)\n",
	ioClip(ioreq)->io_Length,
	((UBYTE *)ioClip(ioreq)->io_Data)[0],
	((UBYTE *)ioClip(ioreq)->io_Data)[1],
	((UBYTE *)ioClip(ioreq)->io_Data)[2],
	((UBYTE *)ioClip(ioreq)->io_Data)[3],
	((UBYTE *)ioClip(ioreq)->io_Data)[0],
	((UBYTE *)ioClip(ioreq)->io_Data)[1],
	((UBYTE *)ioClip(ioreq)->io_Data)[2],
	((UBYTE *)ioClip(ioreq)->io_Data)[3]));

	ioClip(ioreq)->io_Actual = Write(unit->cu_clipFile, ioClip(ioreq)->io_Data, ioClip(ioreq)->io_Length);

	if ((LONG)ioClip(ioreq)->io_Actual == -1)
	{
		D(bug("clipboard.device/writeCb: write failed\n"));
		Close(unit->cu_clipFile);
		unit->cu_clipFile = 0;
		D(bug("clipboard.device/writeCb: releasing semaphore [me=0x%08lx]\n", FindTask(NULL)));
		ReleaseSemaphore(&unit->cu_UnitLock);
		ioClip(ioreq)->io_Error = IOERR_ABORTED;
		ioClip(ioreq)->io_Actual = 0;
		ioClip(ioreq)->io_ClipID = -1;
	}
	else
	{
		ioClip(ioreq)->io_Offset += ioClip(ioreq)->io_Actual;
		if (ioClip(ioreq)->io_Offset > unit->cu_clipSize)
		{
			unit->cu_clipSize = ioClip(ioreq)->io_Offset;
		}
	}
}

/****************************************************************************************/

static void updateCb(struct IORequest *ioreq, struct ClipboardBase *CBBase)
{
	struct ClipboardUnit *unit = (APTR) ioreq->io_Unit;

	if (unit->cu_WriteID != 0 && unit->cu_WriteID == ioClip(ioreq)->io_ClipID)
	{
		D(bug("clipboard.device/updateCb: Closing ClipFile\n"));

		if (unit->cu_clipFile)
		{
			BOOL fileok = FALSE;
			ULONG finalsize;
#pragma pack(2)
			struct
			{
				ULONG id;
				ULONG len;
			} iffh;
#pragma pack()
			do
			{
				if (Seek(unit->cu_clipFile, 0, OFFSET_BEGINNING) == -1) break;
				if (Read(unit->cu_clipFile, &iffh, sizeof(iffh)) != sizeof(iffh)) break;
				LE_SWAPLONG_P(&iffh.id);
				LE_SWAPLONG_P(&iffh.len);
				if ((iffh.id != ID_FORM && iffh.id != ID_CAT && iffh.id != ID_LIST) ||
				    iffh.len > (ULONG)(MAXCLIPSIZE - sizeof(iffh))) break;
				finalsize = iffh.len + sizeof(iffh);

				if (unit->cu_clipSize < finalsize)
				{
					if (Seek(unit->cu_clipFile, unit->cu_clipSize, OFFSET_BEGINNING) == -1) break;
					fileok = !writezero(ioreq,  finalsize - unit->cu_clipSize, CBBase);
				}
				else
				{
					fileok = TRUE;
				}
			} while (0);

			Close(unit->cu_clipFile);
			unit->cu_clipFile = 0;

			if (!fileok)
			{
				DeleteFile(unit->cu_clipFilename);
			}
		}

		if (unit->cu_PostRequesters.mlh_Head->mln_Succ != NULL)
		{
			/* Wake up first reader */
			D(bug("clipboard.device/updateCb: Waking up 0x%08lx\n", ((struct PostRequest*) unit->cu_PostRequesters.mlh_Head)->pr_Waiter));
			Signal(((struct PostRequest*) unit->cu_PostRequesters.mlh_Head)->pr_Waiter, SIGF_SINGLE);
		}
		D(bug("clipboard.device/updateCb: calling ReleaseSemaphore [me=0x%08lx]\n", FindTask(NULL)));

		ReleaseSemaphore(&unit->cu_UnitLock);

		D(bug("clipboard.device/updateCb: Calling monitoring hooks\n"));

		/* Call monitoring hooks. */
		{
			struct Hook        *tnode;
			struct ClipHookMsg  chmsg;

			chmsg.chm_Type      = 0;
			chmsg.chm_ChangeCmd = CMD_UPDATE;
			chmsg.chm_ClipID    = ioClip(ioreq)->io_ClipID;

			ObtainSemaphore(&unit->cu_HookListLock);

			ForeachNode(&unit->cu_HookList, tnode)
			{
				D(bug("Calling hook 0x%08x\n", tnode));
				CallHookA(tnode, unit, &chmsg);
			}
			D(bug("Done\n"));

			ReleaseSemaphore(&unit->cu_HookListLock);
		}

		D(bug("clipboard.device/updateCb: Called monitoring hooks\n"));
	}
	else
	{
		ioClip(ioreq)->io_Error = IOERR_ABORTED;
		ioClip(ioreq)->io_Actual = 0;
	}

	ioClip(ioreq)->io_ClipID = -1;

	D(bug("clipboard.device/updateCb: end of function [me=0x%08lx]\n", FindTask(NULL)));
}

/****************************************************************************************/

